﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapTaxonomyToCrm : MapperRelatedBase
    {
        public MapTaxonomyToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {
        }

        public async Task<ppms_providertaxonomy> MapInsert(Taxonomy taxonomy, Account provider)
        {
            // map provider
            var entity = new ppms_providertaxonomy
            {
                Id = Guid.NewGuid(),
                ppms_PrimaryTaxonomy = taxonomy.IsPrimaryTaxonomy
            };

            // Map relationships
            if (!string.IsNullOrEmpty(taxonomy.CodedSpecialty))
            {
                var specialty = GetTaxonomy(taxonomy.CodedSpecialty);
                if (specialty != null) entity.ppms_TaxonomyId = specialty;
            }

            await Task.Run(() => { });

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            return entity;
        }

        public async Task<ppms_providertaxonomy> MapUpdate(Taxonomy entity, Account provider)
        {
            ppms_providertaxonomy newEntity;

            // get entity
            var crmEntity = GetCrmEntity(provider, entity.CodedSpecialty);

            // matching account not found
            if (crmEntity == null) newEntity = await MapInsert(entity, provider);
            else
            {
                newEntity = new ppms_providertaxonomy
                {
                    Id = crmEntity.Id,
                    ppms_providertaxonomyId = crmEntity.ppms_providertaxonomyId,
                    ppms_PrimaryTaxonomy = crmEntity.ppms_PrimaryTaxonomy
                };

                //Map fields
                newEntity.ppms_PrimaryTaxonomy = entity.IsPrimaryTaxonomy;

                // Set owner to CCN
                if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;
            }

            return newEntity;
        }

        private static ppms_providertaxonomy GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_providertaxonomy;
            if (list == null) return null;

            var ppmsProvidertaxonomies = list as ppms_providertaxonomy[] ?? list.ToArray();

            return ppmsProvidertaxonomies.Any() ? ppmsProvidertaxonomies.FirstOrDefault(x => x.ppms_codedspecialty == id) : null;
        }

        private static Taxonomy ConvertEntity<T>(T entity)
        {
            return (Taxonomy)Convert.ChangeType(entity, typeof(Taxonomy));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_ppms_providertaxonomy");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null) account.ppms_account_ppms_providertaxonomy = ConvertEntityList<ppms_providertaxonomy>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_providertaxonomy>();
            var relatedEntities = provider.ppms_account_ppms_providertaxonomy.ToList();
            Taxonomy entity;

            if (relatedEntities != null && relatedEntities.Any())
            {

                // Map schema entities for delete
                foreach (var item in entities)
                {
                    entity = ConvertEntity(item);
                    var matches = relatedEntities.Where(p => p.ppms_codedspecialty == entity.CodedSpecialty);
                    list.AddRange(matches);
                }
            }

            return EntityDelete((IEnumerable<ppms_providertaxonomy>)list);
        }

        public static IList<Taxonomy> GetNewItems(IList<Taxonomy> entities, Account parent)
        {
            var list = parent.ppms_account_ppms_providertaxonomy;
            if (list == null) return null;

            var newItems = from item in entities
                           where !list.Any(x => x.ppms_codedspecialty == item.CodedSpecialty)
                           select item;

            return newItems.ToList();
        }
    }
}